import org.junit.jupiter.api.*;
import static org.junit.jupiter.api.Assertions.*;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.junit.jupiter.params.provider.ValueSource;

@TestMethodOrder(MethodOrderer.OrderAnnotation.class) // Tests will run in a specific order
class KaprekarRoutineTest {

    private KaprekarRoutine kaprekar;

    @BeforeAll
    static void initAll() {
        System.out.println(">>> Starting KaprekarRoutine tests...");
    }

    @AfterAll
    static void tearDownAll() {
        System.out.println(">>> All tests completed.");
    }

    @BeforeEach
    void init() {
        kaprekar = new KaprekarRoutine();
    }

    @AfterEach
    void tearDown() {
        kaprekar = null;
    }

    // ---------- Valid numbers ----------

    @Test
    @Order(1)
    @DisplayName("Should return 0 steps for 6174")
    void testAlreadyKaprekarConstant() {
        assertEquals(0, kaprekar.stepsToKaprekar(6174));
    }

    @Test
    @Order(2)
    @DisplayName("Number 3524 should reach 6174 in 3 steps")
    void testNumber3524() {
        assertEquals(3, kaprekar.stepsToKaprekar(3524));
    }

    @Test
    @Order(3)
    @DisplayName("Number 2111 should reach 6174 in 5 steps")
    void testNumber2111() {
        assertEquals(5, kaprekar.stepsToKaprekar(2111));
    }

    @RepeatedTest(2)
    @Order(4)
    @DisplayName("Number 1234 should consistently reach 6174 within 7 steps")
    void testNumber1234Repeated() {
        int steps = kaprekar.stepsToKaprekar(1234);
        assertTrue(steps > 0 && steps <= 7);
    }

    // ---------- Invalid inputs ----------

    @Test
    @Order(5)
    @DisplayName("Should throw for numbers < 1000")
    void testTooSmallNumber() {
        assertThrows(IllegalArgumentException.class, () -> kaprekar.stepsToKaprekar(999));
    }

    @Test
    @Order(6)
    @DisplayName("Should throw for numbers > 9999")
    void testTooLargeNumber() {
        assertThrows(IllegalArgumentException.class, () -> kaprekar.stepsToKaprekar(10000));
    }

    @Test
    @Order(7)
    @DisplayName("Should throw for numbers with identical digits like 1111")
    void testAllDigitsSame() {
        assertThrows(IllegalArgumentException.class, () -> kaprekar.stepsToKaprekar(1111));
    }

    // ---------- Edge cases ----------

    @Test
    @Order(8)
    @Tag("edge")
    @DisplayName("Number with leading zeros (e.g. 1000) should work")
    void testNumberWithLeadingZeros() {
        int steps = kaprekar.stepsToKaprekar(1000);
        assertTrue(steps > 0 && steps <= 7);
    }

    @Test
    @Order(9)
    @Disabled("Demonstration: test disabled intentionally")
    @DisplayName("Disabled test example")
    void disabledTest() {
        fail("This test is disabled and should not run.");
    }

    @ParameterizedTest
    @Order(10)
    @ValueSource(ints = {3524, 2111, 1234, 1000})
    @DisplayName("Parameterized test: all numbers should reach 6174 in ≤ 7 steps")
    void testKaprekarWithinSevenSteps(int number) {
        int steps = kaprekar.stepsToKaprekar(number);
        assertTrue(steps <= 7);
    }

    @ParameterizedTest(name = "Number {0} should reach 6174 in {1} steps")
    @CsvSource({
            "3524, 3",
            "2111, 5",
            "6174, 0",
            "1234, 3"
    })
    void testKaprekarStepsWithCsv(int number, int expectedSteps) {
        assertEquals(expectedSteps, kaprekar.stepsToKaprekar(number));
    }
}